/**
  ******************************************************************************
  * @file    py32f410_hal_opa.h
  * @author  MCU Application Team
  * @brief   Header file of OPA HAL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef PY32F410_HAL_OPA_H
#define PY32F410_HAL_OPA_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32f410_hal_def.h"

/** @addtogroup PY32F410_HAL_Driver
  * @{
  */

/** @addtogroup OPA
  * @{
  */

/* Exported types ------------------------------------------------------------*/
/** @defgroup OPA_Exported_Types OPA Exported Types
  * @{
  */

/**
  * @brief  OPA Init structure definition
  */

typedef struct
{
  uint32_t OutputSelect;             /*!< Specifies the OutputSelect.
                                             This parameter must be a value of @ref OPA_OutputSelect */

  uint32_t Mode;                     /*!< Specifies the OPA mode
                                             This parameter must be a value of @ref OPA_Mode
                                             mode is either OPA, COMP */
  uint32_t DigitalFilter;            /*!< Specifies the digital filter. the filter is prohibited 
                                             when the value is zero.
                                             This parameter must be a number between 0 and 0xFFFF */

  uint32_t ITMode;                   /*!< Specify whether the COMP configured in interrupt or polling mode.
                                             This parameter can be set to ENABLE or DISABLE */
} OPA_InitTypeDef;

/**
  * @brief  HAL State structures definition
  */

typedef enum
{
  HAL_OPA_STATE_RESET               = 0x00000000UL, /*!< OPA is not yet Initialized          */
  HAL_OPA_STATE_READY               = 0x00000001UL, /*!< OPA is initialized and ready for use */
  HAL_OPA_STATE_BUSY                = 0x00000004UL, /*!< OPA is enabled and running in normal mode */
  HAL_OPA_STATE_BUSYLOCKED          = 0x00000005UL, /*!< OPA control register is locked
                                                         only system reset allows reconfiguring the opa. */

} HAL_OPA_StateTypeDef;

/**
  * @brief OPA Handle Structure definition
  */
#if (USE_HAL_OPA_REGISTER_CALLBACKS == 1)
typedef struct __OPA_HandleTypeDef
#else
typedef struct
#endif
{
  OPA_TypeDef       *Instance;                    /*!< OPA instance's registers base address   */
  OPA_InitTypeDef   Init;                         /*!< OPA required parameters */
  HAL_StatusTypeDef Status;                         /*!< OPA peripheral status   */
  HAL_LockTypeDef   Lock;                           /*!< Locking object          */
  __IO HAL_OPA_StateTypeDef  State;               /*!< OPA communication state */

#if (USE_HAL_OPA_REGISTER_CALLBACKS == 1)
  void (* MspInitCallback)(struct __OPA_HandleTypeDef *hopa);
  void (* MspDeInitCallback)(struct __OPA_HandleTypeDef *hopa);
#endif /* USE_HAL_OPA_REGISTER_CALLBACKS */

} OPA_HandleTypeDef;

#if (USE_HAL_OPA_REGISTER_CALLBACKS == 1)
/**
  * @brief  HAL OPA Callback ID enumeration definition
  */
typedef enum
{
  HAL_OPA_MSPINIT_CB_ID                     = 0x01UL,  /*!< OPA MspInit Callback ID           */
  HAL_OPA_MSPDEINIT_CB_ID                   = 0x02UL,  /*!< OPA MspDeInit Callback ID         */
  HAL_OPA_ALL_CB_ID                          = 0x03UL   /*!< OPA All ID                        */
} HAL_OPA_CallbackIDTypeDef;

/**
  * @brief  HAL OPA Callback pointer definition
  */
typedef void (*pOPA_CallbackTypeDef)(OPA_HandleTypeDef *hopa);
#endif /* USE_HAL_OPA_REGISTER_CALLBACKS */
/**
  * @}
  */

/* Exported constants --------------------------------------------------------*/
/** @defgroup OPA_Exported_Constants OPA Exported Constants
  * @{
  */
/** @defgroup OPA_OutputSelect OPA OutputSelect
  * @{
  */
#define OPA_OUTPUTSELECT_GPIO            (0x00000000UL)             /*!< output to gpio */
#define OPA_OUTPUTSELECT_ADC             (OPA_CSR_OPAINTOEN)        /*!< output to adc  */
/**
  * @}
  */

/** @defgroup OPA_Mode OPA Mode
  * @{
  */
#define OPA_MODE_OPA              (0x00000000UL)             /*!< Mode OPA */
#define OPA_MODE_COMP             (OPA_CSR_OPA_CMP_CR)       /*!< Mode COMP  */
/**
  * @}
  */

/** @defgroup OPA_OutputLevel OPA Output Level
  * @{
  */
/* Note: Comparator output level values are fixed to "0" and "1",             */
/* corresponding OPA register bit is managed by HAL function to match        */
/* with these values (independently of bit position in register).             */

/* When output polarity is not inverted, comparator output is low when
   the input plus is at a lower voltage than the input minus */
#define OPA_OUTPUT_LEVEL_LOW              (0x00000000UL)
/* When output polarity is not inverted, comparator output is high when
   the input plus is at a higher voltage than the input minus */
#define OPA_OUTPUT_LEVEL_HIGH             (0x00000001UL)

/**
  * @}
  */

/**
  * @}
  */
/* Private constants ---------------------------------------------------------*/
/** @defgroup OPA_Private_Constants OPA Private Constants
  * @brief   OPA Private constants and defines
  * @{
  */

/**
  * @}
  */


/* Private macro -------------------------------------------------------------*/

/** @defgroup OPA_Private_Macros OPA Private Macros
  * @{
  */


/**
  * @}
  */


/* Exported macros -----------------------------------------------------------*/
/** @defgroup OPA_Exported_Macros OPA Exported Macros
  * @{
  */

/** @brief Reset OPA handle state
  * @param  __HANDLE__ OPA handle.
  * @retval None
  */
#define __HAL_OPA_RESET_HANDLE_STATE(__HANDLE__) ((__HANDLE__)->State = HAL_OPA_STATE_RESET)

/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/
/** @defgroup OPA_Exported_Functions OPA Exported Functions
  * @{
  */

/** @defgroup OPA_Exported_Functions_Group1 Initialization and de-initialization functions
  * @{
  */

/* Initialization/de-initialization functions  **********************************/
HAL_StatusTypeDef HAL_OPA_Init(OPA_HandleTypeDef *hopa);
HAL_StatusTypeDef HAL_OPA_DeInit(OPA_HandleTypeDef *hopa);
void HAL_OPA_MspInit(OPA_HandleTypeDef *hopa);
void HAL_OPA_MspDeInit(OPA_HandleTypeDef *hopa);
/**
  * @}
  */


/** @defgroup OPA_Exported_Functions_Group2 Input and Output operation functions
  * @{
  */

/* I/O operation functions  *****************************************************/
HAL_StatusTypeDef HAL_OPA_Start(OPA_HandleTypeDef *hopa);
HAL_StatusTypeDef HAL_OPA_Stop(OPA_HandleTypeDef *hopa);
void              HAL_OPA_IRQHandler(OPA_HandleTypeDef *hopa);
/**
  * @}
  */

/* Peripheral Control functions  ************************************************/
/** @addtogroup OPA_Exported_Functions_Group3
  * @{
  */
uint32_t          HAL_OPA_GetOutputLevel(OPA_HandleTypeDef *hopa);
/* Callback in interrupt mode */
void              HAL_OPA_TriggerCallback(OPA_HandleTypeDef *hopa);
/**
  * @}
  */

/**
  * @}
  */

/** @defgroup OPA_Exported_Functions_Group3 Peripheral Control functions
  * @{
  */

/* Peripheral Control functions  ************************************************/
#if (USE_HAL_OPA_REGISTER_CALLBACKS == 1)
/* OPA callback registering/unregistering */
HAL_StatusTypeDef HAL_OPA_RegisterCallback(OPA_HandleTypeDef *hopa, HAL_OPA_CallbackIDTypeDef CallbackId,
                                             pOPA_CallbackTypeDef pCallback);
HAL_StatusTypeDef HAL_OPA_UnRegisterCallback(OPA_HandleTypeDef *hopa, HAL_OPA_CallbackIDTypeDef CallbackId);
#endif /* USE_HAL_OPA_REGISTER_CALLBACKS */

HAL_StatusTypeDef HAL_OPA_Lock(OPA_HandleTypeDef *hopa);

/**
  * @}
  */

/** @defgroup OPA_Exported_Functions_Group4 Peripheral State functions
  * @{
  */

/* Peripheral State functions  **************************************************/
HAL_OPA_StateTypeDef HAL_OPA_GetState(OPA_HandleTypeDef *hopa);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* PY32F410_HAL_OPA_H */

